# -*- Mode: perl; tab-width: 4; indent-tabs-mode: nil; -*-
#
# Hixie's Web log system
#
# Copyright (c) 2002 by Ian Hickson
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package Log;
use strict;
use vars qw(@ISA);
use PLIF::Service;
use PLIF::DataSource;
@ISA = qw(PLIF::Service PLIF::DataSource);
1;

# XXX duplication of code is starting... I need to factor stuff out

sub provides {
    my $class = shift;
    my($service) = @_;
    return ($service eq 'dispatcher.output.generic' or 
            $service eq 'dispatcher.output' or 
            $service eq 'dispatcher.commands' or
            $service eq 'dataSource.log' or
            $service eq 'dataSource.configuration.client' or
            $service eq 'component.log' or
            $service eq 'app.URIInterpreter' or
            $service eq 'setup.install' or
            $service eq 'setup.configure' or
            $class->SUPER::provides($service));
}

# dataSource.log
sub databaseName {
    return 'default';
}

# dataSource.log
sub databaseType {
    return qw(mysql);
}

__DATA__

sub init {
    my $self = shift;
    my($app) = @_;
    $self->SUPER::init(@_);
    $self->{name} = 'Unnamed Web log';
    $self->{canonicalURI} = 'http://log.example.org/';
    $self->{description} = '';
    eval {
        $app->getService('dataSource.configuration')->getSettings($app, $self, 'log');
    };
}

sub cmdIndex {
    my $self = shift;
    my($app) = @_;
    my $times = $app->getService('dataSource.log')->getTimes($app, 0);
    my $now = time();
    foreach my $time (@$times) {
        $time->{'age'} = $now - $time->{'time'};
    }
    $app->output->index($times);
}

sub outputIndex {
    my $self = shift;
    my($app, $output, $times) = @_;
    my $now = time();
    $output->output('index', {
        'now' => $now,
        'times' => $times,
    });
}

sub cmdShowLog {
    my $self = shift;
    my($app, $start, $count, $order) = @_;
    my $now = time();
    # get the settings from the arguments if we aren't being forced to
    # use specific settings
    $start ||= $app->input->peekArgument('start');
    $count ||= $app->input->peekArgument('count');
    $order ||= $app->input->peekArgument('order');
    # if we are still without settings, it must be the index page
    my $index = not ($start or $count or $order);
    # in which case, we should use the defaults
    $start ||= $now; # default to now
    $count ||= 5; # default to last n posts
    $order ||= -1;
    # get the data from the database
    my $dataSource =  $app->getService('dataSource.log');
    my $posts = $dataSource->getPosts($app, $start, $count);
    my $times = $dataSource->getTimes($app, $start);
    # add age in seconds to each stamp/time
    foreach my $post (@$posts) {
        $post->{'age'} = $now - $post->{'stamp'};
    }
    foreach my $time (@$times) {
        $time->{'age'} = $now - $time->{'time'};
    }
    # and use it all to output our ideas
    $app->output->log($index, $start, $count, $order, $posts, $times);
}

sub outputLog {
    my $self = shift;
    my($app, $output, $index, $start, $count, $order, $posts, $times) = @_;
    my $now = time();
    $output->output('log', {
        'now' => $now,
        'start' => $start,
        'count' => $count,
        'order' => $order,
        'posts' => $posts,
        'index' => $index,
        'times' => $times,
    });
}

# this requires you to be logged in
sub cmdAddLog {
    my $self = shift;
    my($app) = @_;
    my $user = $app->getService('user.login')->hasRight($app, 'poster');
    if (defined($user)) {
        # update the website
        my $title = $app->input->getArgument('title');
        my $content = $app->input->getArgument('content');
        my $stamp = time();
        $app->getService('dataSource.log')->addPost($app, $title, $content, $stamp);
        # update user
        $app->dispatch('showLog', $stamp, 1);
        # send pingbacks
        $self->sendPingbacks($app, $stamp, $content);
        # update weblogs.com
        $self->updateWeblogsCom($app);
        $self->updateRSS($app);
    }
}

# this requires you to be logged in
sub cmdEditLog {
    my $self = shift;
    my($app) = @_;
    my $user = $app->getService('user.login')->hasRight($app, 'poster');
    if (defined($user)) {
        # update database
        my $id = $app->input->getArgument('id');
        my $title = $app->input->getArgument('title');
        my $content = $app->input->getArgument('content');
        my $dataSource = $app->getService('dataSource.log');
        my $stamp = $dataSource->getStampByID($app, $id);
        $self->assert(defined($stamp), 1, 'Could not find entry to be edited');
        $dataSource->editPost($app, $id, $title, $content);
        # update user
        $app->dispatch('showLog', $stamp, 1);
        $self->updateRSS($app);
    }
}

# this requires you to be logged in
sub cmdDeletePingback {
    my $self = shift;
    my($app) = @_;
    my $user = $app->getService('user.login')->hasRight($app, 'poster');
    if (defined($user)) {
        # update database
        my $id = $app->input->getArgument('id');
        my $stamp = $app->getService('dataSource.log')->deletePingback($app, $id);
        $self->assert($stamp, 1, 'Pingback does not exist');
        # update user
        $app->dispatch('showLog', $stamp, 1);
    }
}

sub updateWeblogsCom {
    my $self = shift;
    eval {
        require SOAP::Lite; import SOAP::Lite;
        my $result = SOAP::Lite
            -> service('file:Log/weblogsCom.wsdl')
                -> ping($self->{name}, $self->{canonicalURI});
        if ($result->{'flerror'}) {
            die($result->{'message'});
        }
    };
    if ($@) {
        $self->warn(1, 'Failed to update weblogs.com: '.$@);
    }
}

# http://www.hixie.ch/specs/pingback/pingback
sub sendPingbacks {
    my $self = shift;
    my($app, $stamp, $content) = @_;
    my $www = $app->getService('service.www');
    # make up permalink
    my $permalink = $self->{canonicalURI} . "?start=$stamp&count=1";
    # scan $content for links
    my $links = {};
    while ($content =~ s/href=\"([^\"]+)\"//os) {
        $links->{$www->unescapeHTML($1)}++;
        # using a hash instead of an array avoids duplicates
    }
    foreach my $link (keys(%$links)) {
        # fetch the page
        my($page, $headers) = $www->get($app, $link, $permalink);
        # scan for a pingback link
        my $pingbackServer;
        if (my @pingbackServers = $headers->header('X-Pingback')) {
            # XXX check that there is only one?
            $pingbackServer = $pingbackServers[0];
        } elsif ($page =~ m/<link\s+rel=\"pingback\"\s+href=\"([^\"]+)\"\s*\/?>/os) {
            $pingbackServer = $www->unescapeHTML($1);
        } else {
            $self->dump(4, "There is no pingback server at '$link'");
            next;
        }
        # send pingback
        require RPC::XML::Client; import RPC::XML::Client;
        my $client = RPC::XML::Client->new($pingbackServer);
        my $response = $client->send_request('pingback.ping', $permalink, $link);
        if (not ref $response) {
            $self->dump(4, "Failed to ping back '$pingbackServer': $response");
        } else {
            $self->dump(4, "Got a response from '$pingbackServer': ", $response->as_string);
        }
    }
}

sub updateRSS {
    my $self = shift;
    my($app) = @_;
    my $start = time();
    my $count = 10;
    my $posts = $app->getService('dataSource.log')->getPosts($app, $start, $count);
    my $output1 = $app->output->getOutput('rss', {
        'start' => $start,
        'count' => $count,
        'posts' => $posts,
        'encoding' => 'plain',
    });
    my $output2 = $app->output->getOutput('rss', {
        'start' => $start,
        'count' => $count,
        'posts' => $posts,
        'encoding' => 'html',
    });
    my $htaccess = '';
    $htaccess .= $self->writeOut($output1);
    $htaccess .= $self->writeOut($output2);
    local *FILE;
    open(FILE, '>rss/.htaccess');
    print FILE $htaccess;
    close(FILE);
}

sub writeOut {
    my $self = shift;
    my($output) = @_;
    my $filename = $output->{'data'}->{'encoding'};
    $output->{'string'} =~ s/^Status: 200 OK\nContent-Type: (.+?)\n\n//os;
    local *FILE;
    open(FILE, ">rss/$filename");
    print FILE $output->{'string'};
    close(FILE);
    return "<files $filename>\n  ForceType $1\n</files>\n\n";
}

sub cmdRSS {
    my $self = shift;
    my($app) = @_;
    $app->output->redirect("$self->{canonicalURI}rss/html");
    return;

    # my $start = time();
    # my $count = 10;
    # # get the data from the database
    # my $dataSource =  $app->getService('dataSource.log');
    # my $posts = $dataSource->getPosts($app, $start, $count);
    # # and use it all to output our ideas
    # $app->output->RSS($start, $count, $posts);
}

# # dispatcher.output
# sub outputRSS {
#     my $self = shift;
#     my($app, $output, $start, $count, $posts) = @_;
#     $output->output('rss', {
#         'start' => $start,
#         'count' => $count,
#         'posts' => $posts,
#     });
# }

# app.URIInterpreter
sub interpretURI {
    my $self = shift;
    my($app, $uri) = @_;
    my $dataSource = $app->getService('dataSource.log');
    my $uriTitle = $app->name;
    if ($uri =~ m/[&?]start=([0-9]+)(?:&.*)?$/s) {
        (undef, $uriTitle, undef) = $dataSource->getPostByStamp($app, $1);
    }
    return $uriTitle;
}

# dispatcher.commands
sub cmdPingback_ping {
    # XML RPC call pingback.ping(source, target) : string
    # http://www.hixie.ch/specs/pingback/pingback
    my $self = shift;
    my($app) = @_;
    my $source = $app->input->getArgument('1');
    my $target = $app->input->getArgument('2');
    my $base = quotemeta($self->{canonicalURI});
    my $dataSource = $app->getService('dataSource.log');
    my($stamp, $postID, $targetTitle, $targetContent);
    if (not $target =~ m/^$base\?start=([0-9]+)&count=1$/s or
        not $stamp = $1 or
        not (($postID, $targetTitle, $targetContent) = $dataSource->getPostByStamp($app, $stamp))) {
        $self->dump(1, "Failed pingback attempt.", "Source URI: $source", "Target URI: $target");
        $self->error(1, "The given URI ($target) is not the permalink of a blog entry on this site.");
    }
    # Get information about the source
    my $www = $app->getService('service.www');
    my $sourceContent = $www->get($app, $source);
    my $sourceTitle;
    if ($sourceContent =~ m/<title>(.*?)<\/title>/osi) {
        $sourceTitle = $www->unescapeHTML($1);
    }
    $sourceContent = $www->unescapeHTML($sourceContent); # that way we find the URI however it is hidden away
    my $targetRegexp = $target;
    $targetRegexp =~ s/([^a-zA-Z0-9&#])/\\$1/gos;
    $targetRegexp =~ s/#.*$//gos;
    if (not $sourceContent =~ /$targetRegexp/si) {
        $self->dump(1, "Failed pingback attempt.", "Source URI: $source", "Target URI: $target");
        $self->error(1, "The given URI ($source) does not appear to link to this entry ($target). Please make sure your link is correctly escaped, etc. If you are sure you have made no error, contact me.");
    }
    # store it in the database
    $dataSource->addPingback($app, $postID, $source, $sourceTitle);
    # send e-mail
    eval {
        $app->output('email', $app->getService('user.factory')->getUserByID($app, 1))->newPingback($source, $sourceTitle, $sourceContent, $target, $targetTitle, $targetContent, $postID, $stamp);
    };
    if ($@) {
        $self->warn(4, "Error while sending e-mail about a new pingback: $@");
    }
    $app->output->acknowledge("Thanks for the heads-up!\nsource uri: $source\nsource title: $sourceTitle\ntarget post stamp: $stamp\ntarget title: $targetTitle");
}

# dispatcher.commands
sub cmdPingback_extensions_getPingbacks {
    my $self = shift;
    $self->cmdPingback_getPingbacks(@_);
}

# dispatcher.commands
sub cmdPingback_getPingbacks {
    # XML RPC call pingback.getPingbacks(target) : array of strings
    my $self = shift;
    my($app) = @_;
    my $target = $app->input->getArgument('1');
    my $base = quotemeta($self->{canonicalURI});
    my $dataSource = $app->getService('dataSource.log');
    my($stamp, $postID);
    if (not $target =~ m/^$base\?start=([0-9]+)&count=1$/s or
        not $stamp = $1 or
        not $postID = $dataSource->getIDByStamp($app, $stamp)) {
        $self->error(1, "The given URI ($target) is not the permalink of a blog entry on this site.");
    }
    # get pingbacks from database
    my $uris = $dataSource->getPingbacks($app, $postID);
    $app->output->acknowledge($uris);
}

# dispatcher.output
sub outputNewPingback {
    my $self = shift;
    my($app, $output, $source, $sourceTitle, $sourceContent, $target, $targetTitle, $targetContent, $postID, $stamp) = @_;
    $output->output('newPingback', {
        # content is casually dropped here...
        'source' => { 'uri' => $source, 'title' => $sourceTitle },
        'target' => { 'uri' => $target, 'title' => $targetTitle, 'postID' => $postID, 'stamp' => $stamp },
    });
}

# dispatcher.output
sub strings {
    return (
            'log' => 'The main log screen.',
            'rss' => 'The RSS feed.',
            'newPingback' => 'An e-mail to send to the admin when a pingback is reported.',
            );
}

# dataSource.log
sub getPosts {
    my $self = shift;
    my($app, $from, $count) = @_;
    $self->assert($count !~ m/[^0-9]/os, 1, 'Internal error: count is nonnumeric');
    my $posts = $self->database($app)->execute("SELECT postID, title, content, stamp FROM posts
                                                WHERE stamp <= ? ORDER BY stamp DESC LIMIT $count", $from)->rows;
    foreach my $post (@$posts) {
        my $description = $post->[2];
        $description =~ s/<[^>]*>//gos;
        $description =~ s/\s+/ /gos;
        my $content = $post->[2];
        $content =~ s/<\?stamp\?>/$post->[3]/gos;
        my $pingbacks = $self->database($app)->execute('SELECT pingbackID, source, title FROM pingbacks
                                                        WHERE postID = ?
                                                        ORDER BY pingbackID', $post->[0])->rows;
        foreach my $pingback (@$pingbacks) {
            $pingback->[2] =~ s/\xBB/-/gos; # XXX HACK HACK EVIL HORRIBLE HACK
            # (^ for some reason we end up storing ISO-8859-1 in the database.
            # this converts the most common such characters into US-ASCII.)
            $pingback = {
                'id' => $pingback->[0],
                'uri' => $pingback->[1],
                'title' => $pingback->[2],
            };
        }
        $post = {
            'postID' => $post->[0],
            'title' => $post->[1],
            'originalContent' => $post->[2],
            'content' => $content,
            'description' => $description,
            'stamp' => $post->[3], 
            'pingbacks' => $pingbacks,
        };
    }
    return $posts;
}

# dataSource.log
sub getTimes {
    my $self = shift;
    my($app, $center) = @_;
    my $times = $self->database($app)->execute('SELECT stamp, title FROM posts ORDER BY stamp DESC')->rows;
    my $position = 0;
    my $index = 0;
    foreach my $time (@$times) {
        $time = {
            'time' => $time->[0],
            'title' => $time->[1],
        };
        if ($time->{'time'} < $center) {
            $time->{'position'} = ++$position;
        } else {
            ++$index;
        }
    }
    foreach my $time (@$times) {
        if (--$index <= 0) {
            $time->{'position'} = 0;
            last;
        }
        $time->{'position'} = -1 * $index;
    }
    return $times;
}

# dataSource.log
sub addPost {
    my $self = shift;
    my($app, $title, $content, $stamp) = @_;
    return $self->database($app)->execute('INSERT INTO posts SET title=?, content=?, stamp=?', $title, $content, $stamp)->MySQLID;
}

# dataSource.log
sub addPingback {
    my $self = shift;
    my($app, $postID, $source, $title) = @_;
    # try to select the row
    # if it works, it means it is already in.
    if (defined($self->database($app)->execute('SELECT pingbackID FROM pingbacks WHERE postID=? AND source=?', $postID, $source)->row())) {
        return 0;
    }
    # otherwise it means the row isn't there, we can add it.
    $self->database($app)->execute('INSERT INTO pingbacks SET postID=?, source=?, title=?', $postID, $source, $title)->MySQLID;
    return 1;
}

# dataSource.log
sub deletePingback {
    my $self = shift;
    my($app, $pingbackID) = @_;
    my $database = $self->database($app);
    my $stamp = $database->execute('SELECT posts.stamp FROM posts, pingbacks
                                    WHERE pingbacks.pingbackID = ?
                                    AND posts.postID = pingbacks.postID', $pingbackID)->row;
    if (defined($stamp)) {
        $database->execute('DELETE FROM pingbacks WHERE pingbacks.pingbackID = ?', $pingbackID);
    }
    return $stamp;
}

# dataSource.log
sub editPost {
    my $self = shift;
    my($app, $id, $title, $content) = @_;
    $self->assert($id !~ m/[^0-9]/os, 1, 'Internal error: id is nonnumeric');
    return $self->database($app)->execute("UPDATE posts SET title=?, content=? WHERE postID = $id", $title, $content)->rowsAffected;
}

# dataSource.log
sub getStampByID {
    my $self = shift;
    my($app, $id) = @_;
    $self->assert($id !~ m/[^0-9]/os, 1, 'Internal error: id is nonnumeric');
    return $self->database($app)->execute("SELECT stamp FROM posts WHERE postID = $id")->row;
}

# dataSource.log
sub getIDByStamp {
    my $self = shift;
    my($app, $stamp) = @_;
    $self->assert($stamp !~ m/[^0-9]/os, 1, 'Internal error: stamp is nonnumeric');
    return $self->database($app)->execute("SELECT postID FROM posts WHERE stamp = $stamp")->row;
}

# dataSource.log
sub getPostByStamp {
    my $self = shift;
    my($app, $stamp) = @_;
    $self->assert($stamp !~ m/[^0-9]/os, 1, 'Internal error: stamp is nonnumeric');
    return $self->database($app)->execute("SELECT postID, title, content FROM posts WHERE stamp = $stamp")->row;
}

# dataSource.log
sub getPingbacks {
    my $self = shift;
    my($app, $postID) = @_;
    $self->assert($postID !~ m/[^0-9]/os, 1, 'Internal error: stamp is nonnumeric');
    my $data = $self->database($app)->execute("SELECT source FROM pingbacks WHERE postID = $postID")->rows;
    foreach my $item (@$data) {
        $item = $item->[0];
    }
    return $data;
}

# dataSource.log
sub setupInstall {
    my $self = shift;
    my($app) = @_;
    $app->output->setupProgress('component.log');
    my $helper = $self->helper($app);
    my $database = $self->database($app);

    if (not $helper->tableExists($app, $database, 'posts')) {
        $app->output->setupProgress('dataSource.log.posts');
        $database->execute('
            CREATE TABLE posts (
                                postID integer unsigned auto_increment NOT NULL PRIMARY KEY,
                                title text NOT NULL,
                                content text NOT NULL,
                                stamp integer unsigned NOT NULL DEFAULT 0
                               )
        ');
    } else {
        # check its schema is up to date
    }

    if (not $helper->tableExists($app, $database, 'pingbacks')) {
        $app->output->setupProgress('dataSource.log.pingbacks');
        $database->execute('
            CREATE TABLE pingbacks (
                                pingbackID integer unsigned auto_increment NOT NULL PRIMARY KEY,
                                postID integer unsigned NOT NULL,
                                source TEXT,
                                title text NOT NULL,
                               )
        ');
    } else {
        # check its schema is up to date
        # XXX should change 'source' from VARCHAR(128) to TEXT
        # XXX and then remove the postID index
    }

    # XXX should ask which user to e-mail, and store user id in .PLIF
    # XXX should add pref so that we know what contact method the user wants

    my $userDataSource = $app->getService('dataSource.user');
    $userDataSource->addRight($app, 'poster');
    return;
}

sub setupConfigure {
    my $self = shift;
    my($app) = @_;
    $app->output->setupProgress('component.log.settings');

    # get the name of the log
    my $name = $app->input->getArgument('log.name', $self->{name});
    if (not defined($name)) {
        return 'log.name';
    }
    # set the name of the log
    $self->{name} = $name;

    # get the uri of the log
    my $uri = $app->input->getArgument('log.canonicalURI', $self->{canonicalURI});
    if (not defined($uri)) {
        return 'log.canonicalURI';
    }
    # set the uri of the log
    $self->{canonicalURI} = $uri;

    # get the description of the log
    my $description = $app->input->getArgument('log.description', $self->{description});
    if (not defined($description)) {
        return 'log.description';
    }
    # set the description of the log
    $self->{description} = $description;

    # save the name, uri and description of the log
    $app->getService('dataSource.configuration')->setSettings($app, $self, 'log');
    return;

}

# dataSource.configuration.client
sub settings {
    # if you change this, check out setupConfigure to make sure it is still up to date
    return qw(name canonicalURI description);
}
